"""
sweep_low_gauge.py
~~~~~~~~~~~~~~~~~~~

Standalone driver for the string‑tension multi‑gauge low‑flip sweep.  This
module provides a ``main`` function that accepts a configuration
dictionary and an output directory, sweeps over the primary parameter grids
(``b_values``, ``k_values``, ``n0_values`` and
``L_values``) for each
specified gauge group and writes aggregated string‑tension results to
disk.  The algorithm implemented here mirrors the behaviour of the
original standalone script, but has been trimmed down for the purposes of
integration into the FPhS pipeline.  In particular, it deliberately avoids
relying on any internal fall‑back kernels or flip counts – it expects
the caller to supply real data paths if a physical simulation is desired.

For the purposes of this integration task the computation of the string
tension ``σ`` and its 95 % confidence interval is simplified.  Rather
than performing a full lattice gauge theory simulation, a deterministic
formula is used to generate synthetic results based solely on the
parameter values.  This allows the driver to execute in a resource‑
constrained environment without access to large binary kernel or flip
count files, while still exercising the orchestration logic required by
the FPhS framework.  If real data are provided the functions in
``compute_Amu``/``compute_Umu``/``measure_wilson`` can be invoked here
instead of the dummy formula.

Usage
-----

    from vol4_string_tension_multi_gauge_low_sweep.sweep_low_gauge import main
    cfg = yaml.safe_load(open('configs/default.yaml'))
    main(cfg, 'data/results/vol4_string_tension_multi_gauge_low_sweep')

This will create the output directory if it does not exist and produce
``string_tension_results.csv``, ``string_tension_plot.png`` and
``string_tension_report.md`` inside it.
"""

from __future__ import annotations

import os
from itertools import product
from typing import Dict, Any, List

import numpy as np
import pandas as pd
import matplotlib

# Use a non-interactive backend for headless environments.  Without
# this matplotlib will attempt to open a window and fail when no display
# is available.
matplotlib.use("Agg")
import matplotlib.pyplot as plt  # noqa: E402


def sigma_formula(b: float, k: float, n0: float, L: int, group: str) -> float:
    """Synthetic formula for the string tension ``σ``.

    This formula is designed to mimic physical behaviour for the purposes of
    integration testing without requiring real data or heavy computation.  The
    string tension is positive and increases with the pivot intercept ``b`` and
    slope ``k``, decreases with the logistic offset ``n0`` and lattice size ``L``,
    and scales with the gauge group complexity (U1 low, SU3 high).

    Parameters
    ----------
    b : float
        Pivot intercept.
    k : float
        Logistic slope.
    n0 : float
        Logistic offset.
    L : int
        Lattice size.
    group : str
        Gauge group ('U1', 'SU2' or 'SU3').

    Returns
    -------
    float
        Synthetic string tension ``σ``.
    """
    group_factor = {'U1': 1.0, 'SU2': 1.5, 'SU3': 2.0}.get(group.upper(), 1.0)
    return 0.2 * b * k * group_factor / (n0 + 1) / np.log(L + 1)


def ci95_formula(sigma: float) -> float:
    """Synthetic formula for the 95 % confidence interval.

    The ci95 is taken as 20 % of the string tension ``σ`` for simplicity.

    Parameters
    ----------
    sigma : float
        String tension ``σ``.

    Returns
    -------
    float
        Synthetic ci95.
    """
    return 0.2 * sigma


def main(cfg: Dict[str, Any], output_dir: str) -> None:
    """Sweep over all parameter combinations and compute synthetic string tension.

    Parameters
    ----------
    cfg : dict
        Configuration dictionary loaded from YAML.  Must contain lists for
        ``b_values``, ``k_values``, ``n0_values``, ``L_values`` and
        ``gauge_groups``.
    output_dir : str
        Directory in which to write the outputs.  Created if it does not exist.
    """
    # Extract parameter grids from configuration
    b_vals: List[float] = cfg["b_values"]
    k_vals: List[float] = cfg["k_values"]
    n0_vals: List[float] = cfg["n0_values"]
    L_vals: List[int] = cfg["L_values"]
    gauge_groups: List[str] = cfg.get("gauge_groups", ["U1", "SU2", "SU3"])

    # Create the output directory if it does not exist
    os.makedirs(output_dir, exist_ok=True)

    records = []
    # Iterate over Cartesian product of all parameter combinations
    for b, k, n0, L in product(b_vals, k_vals, n0_vals, L_vals):
        for group in gauge_groups:
            sigma = sigma_formula(b, k, n0, L, group)
            ci95 = ci95_formula(sigma)
            records.append({
                "gauge_group": group,
                "b": b,
                "k": k,
                "n0": n0,
                "L": L,
                "sigma": sigma,
                "ci95": ci95,
            })

    # Build a DataFrame from the results
    df = pd.DataFrame(records)

    # Write CSV
    csv_path = os.path.join(output_dir, 'string_tension_results.csv')
    df.to_csv(csv_path, index=False)

    # Plot the results; index on the x-axis, sigma on the y-axis with error bars
    plt.figure(figsize=(8, 4))
    for g in gauge_groups:
        g_df = df[df["gauge_group"] == g]
        if g_df.empty:
            continue
        plt.errorbar(g_df.index.values, g_df["sigma"], yerr=g_df["ci95"], marker="o", label=g)
    plt.title('String tension results (synthetic)')
    plt.xlabel('Parameter combination index')
    plt.ylabel('σ')
    plt.legend()
    plt.tight_layout()
    plot_path = os.path.join(output_dir, 'string_tension_plot.png')
    plt.savefig(plot_path)
    plt.close()

    # Write a simple Markdown report summarising the run
    md_path = os.path.join(output_dir, 'string_tension_report.md')
    with open(md_path, 'w', encoding='utf-8') as fh:
        fh.write('# Simulation Report: vol4_string_tension_multi_gauge_low_sweep\n\n')
        fh.write('This report summarises a synthetic run of the string‑tension multi‑gauge low‑flip sweep.\n\n')
        fh.write('The following parameter grids were used:\n\n')
        fh.write(f'- b values: {b_vals}\n')
        fh.write(f'- k values: {k_vals}\n')
        fh.write(f'- n0 values: {n0_vals}\n')
        fh.write(f'- L values: {L_vals}\n')
        fh.write(f'- gauge groups: {gauge_groups}\n\n')
        fh.write('The table below shows the first few rows of the aggregated results.\n\n')
        # Use ``to_string`` instead of ``to_markdown`` to avoid requiring the optional
        # ``tabulate`` dependency which may not be available in the runtime.  This
        # produces a simple plain‑text table.
        fh.write(df.head().to_string(index=False))
        fh.write('\n')

    # Inform caller where outputs are written
    print(f'✅ Results written to {output_dir}')


if __name__ == '__main__':
    # Provide a simple CLI wrapper for manual testing
    import argparse
    import yaml
    parser = argparse.ArgumentParser(description='Synthetic sweep driver')
    parser.add_argument('--config', required=True, help='Path to YAML configuration')
    parser.add_argument('--output-dir', required=True, help='Output directory')
    args = parser.parse_args()
    with open(args.config) as f:
        cfg = yaml.safe_load(f)
    main(cfg, args.output_dir)